function [pall] = MainAllTests(nrOfSpecies,nrOfReactionsInternal,nrOfReactionsExchange,nrOfReactionsRemoved,cv,nrOfTests)
% This code implements Case Study II in the manuscript titled "Assessing
% and Resolving Model Misspecifications in Metabolic Flux Analysis".
% More specifically, the code generates a random stoichiometric matrix with
% a full rank, simulates an artificial data vector, remove a random set of
% reactions from the stoichiometric matrix, and performs RESET test, F-test
% and Lagrange multiplier test to detect model misspecification. 
%
% Requires: RESET.m, Ftest.m, LMtest.m, and RBMNToolbox (https://sourceforge.net/projects/rmbntoolbox/)
%
% Written by: Rudiyanto Gunawan
% Date: 14 February 2017

% Parameter definition
% nrOfSpecies: number of metabolic species in the network
% nrOfReactionsInternal: number of intracellular reactions in the true model
% nrOfReactionsExchange: number of exchange fluxes 
% nrOfReactionsRemoved: number of missing reactions  
% cv: coefficient of variation of the external flux data
% nrOfTests: number of independent tests 

pall = [];
for k = 1:nrOfTests
    k
    
    %% Generate a large random biochemical network graph using RBMNToolbox
    % The following implementation of the RBMNToolbox to generate the 
    % random biochemical reaction network was taken from an example given 
    % in this toolbox. 
    reactionConnectionsPDF = [0 50 30 20]; % Reaction's connection probabilities
    
    % Generate a large graph with many more reactions than desired
    G = makeGraphRandom(nrOfSpecies, 10*nrOfSpecies, reactionConnectionsPDF);
    
    % Convert the graph to RMBN structure.
    [RMBN, speciesIcs, reactionIcs] = translateGraphToRMBN(G);
    
    % Find reactions with given number of connections.
    reactionIcs_1 = getReactionIcs( RMBN, 'nrOfConnections', 1 );
    reactionIcs_2 = getReactionIcs( RMBN, 'nrOfConnections', 2 );
    reactionIcs_3 = getReactionIcs( RMBN, 'nrOfConnections', 3 );
    reactionIcs_4 = getReactionIcs( RMBN, 'nrOfConnections', 4 );
       
    % Reactions connecting to 2 species: define the other species as a reactant and the other as a product.
    for reactionIx = reactionIcs_2
        speciesIcs = getSpeciesIcs( RMBN, 'reactionIx', reactionIx );
        order = randperm(2); % Random order: both species have a possibility to be the reactant (or the product).
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(1)), 'reactant' );
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(2)), 'product' );
    end
    
    % Reactions connecting to 3 species: define reactant and product randomly
    for reactionIx = reactionIcs_3
        speciesIcs = getSpeciesIcs( RMBN, 'reactionIx', reactionIx );
        order = randperm(3); % Random order: species have a possibility to be the reactant (or the product).
        
        %Set one of the species as a reactant and one as a product
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(1)), 'reactant' );
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(2)), 'product' );
        
        %Set the remaining species as reactant or product randomly
        if rand(1)<=0.5
            RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(3)), 'reactant' );
        else
            RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(3)), 'product' );
        end
    end
    
    for reactionIx = reactionIcs_4
        speciesIcs = getSpeciesIcs( RMBN, 'reactionIx', reactionIx );
        order = randperm(4); % Random order: species have a possibility to be the reactant (or the product).
        
        %Set one of the species as a reactant and one as a product
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(1)), 'reactant' );
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(2)), 'reactant' );
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(3)), 'product' );
        RMBN = setSpeciesFunction( RMBN, reactionIx, speciesIcs(order(4)), 'product' );
    end
    
    % Assign stoichiometric coefficients for all reactions.
    allReactionIcs = getReactionIcs( RMBN ); % All reactions.
    reactantStoichPDF = [1]; % Reactant stoichiometry probability
    productStoichPDF = [1]; % Product stoichiometry probability
    RMBN = setStoichiometry( RMBN, ...
        allReactionIcs, reactantStoichPDF, productStoichPDF );
    
    % Obtain the stoichiometric matrix of the graph
    Sbig = full(getStoichiometry(RMBN));
    
    %% Generate the stoichiometric matrix of desired dimensions
    % Generate a square invertible stoichiometric matrix
    [A jb] = rref(Sbig);
    Sbig = Sbig(:,jb);
    
    % Generate the reduced stoichiometric matrix    
    rxnIndex = 1:size(Sbig,2);
    spIndex = 1:100;
    Sred = [];  
    while and(rank(Sred) < nrOfReactionsInternal - nrOfReactionsRemoved, length(rxnIndex)~=0)
        numsp = [];
        for j = 1:length(rxnIndex)
            sp = find(abs(Sbig(:,rxnIndex(j))~=0)); %Indeces of species in each remaining reaction
            numsp(j) = length(intersect(sp,spIndex)); %Number of common species between each remaining reaction and the unaccounted set
        end
        [maxval, jmax] = max(numsp);
        Sred = [Sred Sbig(:,rxnIndex(jmax))];
        sp = intersect(find(abs(Sbig(:,rxnIndex(jmax))~=0)),spIndex);
        spIndex = setxor(sp,spIndex);
        rxnIndex = setxor(rxnIndex(jmax),rxnIndex);
    end
    
    % Generate the missing reactions
    Srem = Sbig(:,rxnIndex);
    col = randperm(size(Srem,2),nrOfReactionsRemoved);
    Somit = Srem(:,col);
    Srem = Srem(:,setdiff(1:size(Srem,2),col));
    
    % Generate the true stoichiometric matrix
    S = [Sred Somit];
    
    %% Generate in silico data
    % Compute the basis vectors for the data generate
    [U Sigma V] = svd(S(nrOfSpecies-nrOfReactionsExchange+1:end,:));
    Vbase = V(:,nrOfSpecies-nrOfReactionsExchange+1:end);
    
    % Generate the data using the true stoichiometric matrix
    vI = Vbase*(2*rand(size(Vbase,2),1)-1);
    vEdata = S*vI.*(1+cv*randn(nrOfSpecies,1));
    
    %% Perform tests for model misspecifications 
    
    % F-test and LM test against the true set of missing reactions
    Z = Somit;
    p1 = Ftest(vEdata,Sred,Z);
    p3 = LMtest(vEdata,Sred,Z);
    
    % F-test and LM test against a distinct set of missing reactions
    Z = Srem(:,randperm(size(Srem,2),nrOfReactionsRemoved));
    p2  = Ftest(vEdata,Sred,Z);
    p4 = LMtest(vEdata,Sred,Z);
    
    % RESET test using the reduced stoichiometric matrix using p = 1 and 2
    p5 = RESET(vEdata,Sred,1);
    p6 = RESET(vEdata,Sred,2);
    
    % RESET test using the true stoichiometric matrix using p = 1 and 2
    p7 = RESET(vEdata,S,1);
    p8 = RESET(vEdata,S,2);
     
    pall = [pall; p1 p2 p3 p4 p5 p6 p7 p8];    
end
